/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.sam;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import net.sf.picard.PicardException;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.cmdline.Usage;
import net.sf.picard.io.IoUtil;
import net.sf.picard.sam.IndexingOutputStream;
import net.sf.picard.util.Log;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMFileWriter;
import net.sf.samtools.SAMFileWriterFactory;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMUtils;
import net.sf.samtools.util.BlockCompressedFilePointerUtil;
import net.sf.samtools.util.BlockCompressedInputStream;
import net.sf.samtools.util.BlockCompressedOutputStream;
import net.sf.samtools.util.BlockCompressedStreamConstants;
import net.sf.samtools.util.CloserUtil;
import net.sf.samtools.util.Md5CalculatingOutputStream;
import net.sf.samtools.util.RuntimeIOException;

public class GatherBamFiles
extends CommandLineProgram {
    @Usage
    public final String USAGE = "Concatenates one or more BAM files together as efficiently as possible. Assumes that the list of BAM files provided as INPUT are in the order that they should be concatenated and simply concatenates the bodies of the BAM files while retaining the header from the first file.  Operates via copying of the gzip blocks directly for speed but also supports generation of an MD5 on the output and indexing of the output BAM file. Only support BAM files, does not support SAM files.";
    @Option(shortName="I", doc="One or more BAM files or text files containing lists of BAM files one per line.")
    public List<File> INPUT;
    @Option(shortName="O", doc="The output BAM file to write.")
    public File OUTPUT;
    @Option(doc="Whether to use low-level gzip block copying for performance. The block-copying method is usually 3-5 times faster and is recommended. Non block-copying is supported primarily for testing.")
    public boolean BLOCK_COPY = true;
    private static final Log log = Log.getInstance(GatherBamFiles.class);

    public static void main(String[] args) {
        GatherBamFiles gatherer = new GatherBamFiles();
        gatherer.CREATE_INDEX = true;
        gatherer.instanceMainWithExit(args);
    }

    @Override
    protected int doWork() {
        List<File> inputs = IoUtil.unrollFiles(this.INPUT, ".bam", ".sam");
        for (File f : inputs) {
            IoUtil.assertFileIsReadable(f);
        }
        IoUtil.assertFileIsWritable(this.OUTPUT);
        if (this.BLOCK_COPY) {
            GatherBamFiles.gatherWithBlockCopying(inputs, this.OUTPUT, this.CREATE_INDEX, this.CREATE_MD5_FILE);
        } else {
            GatherBamFiles.gatherNormally(inputs, this.OUTPUT, this.CREATE_INDEX, this.CREATE_MD5_FILE);
        }
        return 0;
    }

    private static void gatherNormally(List<File> inputs, File output, boolean createIndex, boolean createMd5) {
        SAMFileReader tmp = new SAMFileReader(inputs.get(0));
        SAMFileHeader header = tmp.getFileHeader();
        tmp.close();
        SAMFileWriter out = new SAMFileWriterFactory().setCreateIndex(createIndex).setCreateMd5File(createMd5).makeSAMOrBAMWriter(header, true, output);
        for (File f : inputs) {
            log.info("Gathering " + f.getAbsolutePath());
            SAMFileReader in = new SAMFileReader(f);
            for (SAMRecord rec : in) {
                out.addAlignment(rec);
            }
            CloserUtil.close(in);
        }
        out.close();
    }

    private static void gatherWithBlockCopying(List<File> bams, File output, boolean createIndex, boolean createMd5) {
        try {
            OutputStream out = new FileOutputStream(output);
            if (createMd5) {
                out = new Md5CalculatingOutputStream(out, new File(output.getAbsolutePath() + ".md5"));
            }
            if (createIndex) {
                out = new IndexingOutputStream(out, new File(output.getParentFile(), IoUtil.basename(output) + ".bai"));
            }
            boolean isFirstFile = true;
            for (File f : bams) {
                log.info("Gathering " + f.getAbsolutePath());
                FileInputStream in = new FileInputStream(f);
                BlockCompressedInputStream.FileTermination term = BlockCompressedInputStream.checkTermination(f);
                if (term == BlockCompressedInputStream.FileTermination.DEFECTIVE) {
                    throw new PicardException(f.getAbsolutePath() + " does not have a valid GZIP block at the end of the file.");
                }
                if (!isFirstFile) {
                    long vOffsetOfFirstRecord = SAMUtils.findVirtualOffsetOfFirstRecordInBam(f);
                    BlockCompressedInputStream blockIn = new BlockCompressedInputStream(f);
                    blockIn.seek(vOffsetOfFirstRecord);
                    long remainingInBlock = blockIn.available();
                    if (remainingInBlock >= 0L) {
                        BlockCompressedOutputStream blockOut = new BlockCompressedOutputStream(out, null);
                        IoUtil.transferByStream(blockIn, blockOut, remainingInBlock);
                        blockOut.flush();
                    }
                    blockIn.close();
                    for (long pos = BlockCompressedFilePointerUtil.getBlockAddress(blockIn.getFilePointer()); pos > 0L; pos -= in.skip(pos)) {
                    }
                }
                long currentPos = in.getChannel().position();
                long length = f.length();
                long skipLast = term == BlockCompressedInputStream.FileTermination.HAS_TERMINATOR_BLOCK ? (long)BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK.length : 0L;
                long bytesToWrite = length - skipLast - currentPos;
                IoUtil.transferByStream(in, out, bytesToWrite);
                in.close();
                isFirstFile = false;
            }
            ((OutputStream)out).write(BlockCompressedStreamConstants.EMPTY_GZIP_BLOCK);
            ((OutputStream)out).close();
        }
        catch (IOException ioe) {
            throw new RuntimeIOException(ioe);
        }
    }
}

